<?php

namespace Modules\OpenAI\DataTables;

use App\DataTables\DataTable;
use Yajra\DataTables\Html\Column;
use Illuminate\Http\JsonResponse;
use Modules\OpenAI\Entities\Archive;
use Yajra\DataTables\Facades\DataTables;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;

class AiAvatarDataTable extends DataTable
{
    /**
     * Display ajax response
     *
     * @return JsonResponse
     */
    public function ajax(): JsonResponse
    {
        $video = $this->query();

        return DataTables::eloquent($video)
            ->editColumn('name', function ($video) {
                return trimWords($video->title, 40);
            })
            ->editColumn('user_id', function ($video) {
                return '<a href="' . route('users.edit', ['id' => $video->video_creator_id]) . '">' . optional($video->videoCreator)->name . '</a>';
            })

            ->editColumn('created_at', function ($video) {
                return timeZoneFormatDate($video->created_at);
            })
            ->addColumn('action', function ($video) {
                $show = '<a data-bs-toggle="tooltip" data-bs-placement="top" data-bs-original-title="' . __('Show') . '" title="' . __('Show') . '" href="javascript:void(0)" onclick="showVideo(\'' . addslashes($video->videoUrl()) . '\', \'' . addslashes($video->title) . '\')" class="action-icon"><i class="feather icon-eye"></i></a>
                        <div class="modal fade" id="videoModal" tabindex="-1" aria-labelledby="videoModalLabel" aria-hidden="true">
                            <div class="modal-dialog modal-lg">
                                <div class="modal-content">
                                    <div class="modal-header">
                                        <h5 class="modal-title text-start text-to-video-title" id="videoModalLabel"></h5> <!-- Title will be dynamically updated -->
                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                    </div>
                                    <div class="modal-body text-center">
                                        <div id="videoContainer">
                                            <!-- Video will be loaded here -->
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>&nbsp;';
                $download = '<a data-bs-toggle="tooltip" data-bs-placement="top" data-bs-original-title="' . __('Download') . '" title="' . __('Download') . '" href="' . $video->videoUrl() . '" download="'.  str_replace('.', '', $video->title) .'" class="action-icon"><i class="feather icon-download"></i></a>&nbsp;';
                $delete = '<form method="POST" action="' . route('admin.features.ai-avatar.delete', ['id' => $video->id]) . '" id="delete-video-' . $video->id . '" accept-charset="UTF-8" class="display_inline">
                    ' . csrf_field() . '
                    ' . method_field('DELETE') . '
                    <a class="action-icon confirm-delete" type="button" data-id=' . $video->id . ' data-delete="video" data-label="Delete" data-bs-toggle="modal" data-bs-target="#confirmDelete" data-title="' . __('Delete :x', ['x' => __('Video')]) . '" data-message="' . __('Are you sure to delete this?') . '">
                        <span data-bs-toggle="tooltip" data-bs-placement="top" data-bs-original-title="' . __('Delete') . '" title="' . __('Delete') . '">
                            <i class="feather icon-trash"></i>
                        </span> 
                    </button>
                    </form>';

                return $show . $download . $delete;

            })
            ->rawColumns(['image', 'user_id', 'name', 'created_at', 'action'])
            ->make(true);
    }

    /**
     * Get query source of dataTable.
     *
     * @return QueryBuilder
     */
    public function query(): QueryBuilder
    {


        $videos = Archive::with(['metas' => function($query) {
            $query->whereIN('key', ['video_creator_id', 'file_name'])
                ->select(['owner_id', 'key', 'value']);
            }])

            ->leftJoin('archives_meta as meta_creator', function ($join) {
                $join->on('archives.id', '=', 'meta_creator.owner_id')
                    ->where('meta_creator.key', '=', 'video_creator_id');
            })
            ->leftJoin('users as creators', 'meta_creator.value', '=', 'creators.id')

            ->select([
                'archives.id',
                'archives.title',
                'archives.created_at',
                'creators.name as creator_name',
            ])
            ->where('archives.type', 'ai_avatar')
            ->filter('Modules\OpenAI\Filters\v2\VideoFilter');

        return $this->applyScopes($videos);

    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return HtmlBuilder
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('dataTableBuilder')
            ->minifiedAjax()
            ->selectStyleSingle()
            ->columns($this->getColumns())
            ->parameters(dataTableOptions(['dom' => 'Bfrtip']));
    }

    /**
     * Get the dataTable columns definition.
     *
     * @return array
     */
    public function getColumns(): array
    {
        return [
            new Column(['data'=> 'id', 'name' => 'id', 'title' => '', 'visible' => false, 'width' => '0%' ]),
            new Column(['data'=> 'title', 'name' => 'title', 'title' => __('Title'), 'searchable' => true, 'orderable' => true, 'width'=>'45%']),
            (new Column(['data'=> 'user_id', 'name' => 'creators.name', 'title' => __('Creator'), 'orderable' => true, 'searchable' => true, 'width'=>'25%']))->addClass('text-center'),
            (new Column(['data'=> 'created_at', 'name' => 'created_at', 'title' => __('Created At'), 'orderable' => true, 'searchable' => false, 'width'=>'15%']))->addClass('text-center'),
            new Column(['data'=> 'action', 'name' => 'action', 'title' => '', 'width' => '15%', 'visible' => true, 'orderable' => false, 'searchable' => false, 'className' => 'text-right align-middle'])
        ];
    }

}
