<?php

namespace Modules\OpenAI\AiProviders\OpenAi\Resources;

class TextToVideoDataProcessor
{
    private array $data = [];

    public function __construct(array $aiOptions = [])
    {
        $this->data = $aiOptions;
    }

    public function validationRules(): array
    {
        return [];
    }

    public function rules(): array
    {
        return [
            'duration' => [
                'sora-2' => [4, 8, 12],
                'sora-2-pro' => [4, 8, 12],
            ],
            'size' => [
                'sora-2' => ['720x1280', '1280x720'],
                'sora-2-pro' => ['720x1280', '1280x720', '1024x1792', '1792x1024'],
            ],
            'service' => [
                'text-to-video' => [
                    'prompt' => true,
                ],
            ],
        ];
    }

    public function textToVideoOptions(): array
    {
        return [
            [
                'type' => 'checkbox',
                'label' => 'Provider State',
                'name' => 'status',
                'value' => '',
                'visibility' => true,
            ],
            [
                'type' => 'text',
                'label' => 'Provider',
                'name' => 'provider',
                'value' => 'OpenAi',
                'visibility' => true,
            ],
            [
                'type' => 'dropdown',
                'label' => 'Service',
                'name' => 'service',
                'value' => [
                    'text-to-video',
                ],
                'visibility' => true,
                'admin_visibility' => false,
                'default_value' => 'text-to-video',
            ],
            [
                'type' => 'dropdown',
                'label' => 'Models',
                'name' => 'model',
                'value' => [
                    'sora-2',
                    'sora-2-pro',
                ],
                'visibility' => true,
                'required' => true,
                'default_value' => 'sora-2',
            ],
            [
                'type' => 'dropdown',
                'label' => 'Duration',
                'name' => 'duration',
                'value' => [
                    4,
                    8,
                    12,
                ],
                'visibility' => true,
            ],
            [
                'type' => 'dropdown',
                'label' => 'Size',
                'name' => 'size',
                'value' => [
                    '720x1280',
                    '1280x720',
                    '1024x1792',
                    '1792x1024',
                ],
                'default_value' => '720x1280',
                'visibility' => true,
            ]
        ];
    }

    public function customerValidationRules(): array
    {
        $models = 'sora-2,sora-2-pro';

        $validationRules = [
            'prompt' => 'required',
            'provider' => 'required',
            'options.model' => 'required|in:' . $models,
        ];

        $validationMessage = [
            'prompt.required' => __('Please enter a prompt to generate a video.'),
            'provider.required' => __('Please select a provider.'),
            'options.model.required' => __('Please select a model.'),
            'options.model.in' => __('Please select a valid model.'),
        ];

        return [
            $validationRules,
            $validationMessage,
        ];
    }

    public function prepareData(): array
    {
        return $this->getFilteredData();
    }

    public function getFilteredData(): array
    {
        $model = data_get($this->data, 'options.model', 'sora-2');

        $prompt = trim((string) data_get($this->data, 'prompt', ''));
        $size = data_get($this->data, 'options.size', '720x1280');
        $seconds = data_get($this->data, 'options.duration') ?? data_get($this->data, 'options.seconds');

        $payload = [
            'model' => $model,
            'prompt' => $prompt,
            'size' => $size,
        ];

        if (!is_null($seconds)) {
            $payload['seconds'] = (string) $seconds;
        }

        return array_filter($payload, fn ($value) => !is_null($value));
    }

    public function findProviderData(array $data, string $searchKey, bool $returnKey = true, array $options = []): ?string
    {
        foreach ($data as $key => $values) {
            if (array_key_exists($searchKey, $values)) {
                return $returnKey ? $key : $this->formatValue($values[$searchKey], $options);
            }
        }

        return null;
    }

    private function formatValue(string $value, array $options): string
    {
        return $value;
    }
}

